//
//  GeometryGamesBufferPool.swift
//
//  Created by Jeff on 3/10/20.
//  Copyright © 2020 Jeff Weeks. All rights reserved.
//

import MetalKit

//	Apple's Metal Best Practices Guide's section on Persistent Objects,
//	available at
//
//		https://developer.apple.com/library/archive/documentation/3DDrawing/Conceptual/MTLBestPracticesGuide/PersistentObjects.html
//
//	has a subsection on Allocate Resource Storage Up Front,
//	which recommends creating MTLBuffer objects early on,
//	and then re-using those same objects -- with fresh data
//	for each frame of course -- during the animation.
//	[Confession:  I personally don't know why buffer allocation
//	is expensive, but online sources (for Vulkan as well as Metal)
//	all agree that it is.]
//
//	For each "inflight" data buffer, the app keeps
//	a GeometryGamesBufferPool containing several instances
//	of that buffer, so that while the GPU renders a frame
//	using one instance of the buffer, the CPU can already
//	be filling another instance in preparation for the next frame.
//	A GeometryGamesBufferPool creates new buffers as needed,
//	so getBuffer() may fail only in the extremely unlikely situation
//	that a GeometryGamesBufferPool has no more buffers available
//	and has failed in its attempt to create a new one.

class GeometryGamesBufferPool {

	let itsDevice: MTLDevice
	let itsBufferSize: UInt
	let itsBufferLabel: String
	var itsPool = [MTLBuffer]()
	var itsCount = 0
	
	//	Mutable arrays are not thread-safe,
	//	so let's use a lock to serialize access to itsPool.
	let itsPoolLock = NSLock()
	
	init(
		device: MTLDevice,
		bufferSize: UInt,
		bufferLabel: String
	) {
		itsDevice = device
		itsBufferSize = bufferSize
		itsBufferLabel = bufferLabel
	}
	
	func get() -> MTLBuffer {
		
		itsPoolLock.lock()
		
			let theBuffer: MTLBuffer
			if itsPool.isEmpty {
			
				guard let theNewBuffer = itsDevice.makeBuffer(
					length: Int(itsBufferSize),
					options: [MTLResourceOptions.storageModeShared])
				else {
					//	In the unlikely event that itsDevice can't create theNewBuffer,
					//	let's just give up.  On the one hand, this may seem like
					//	a drastic response, but on the other hand, it lets us
					//	return a MTLBuffer instead of a MTLBuffer?, which keep
					//	the caller's code cleaner.  Also, if itsDevice can't
					//	create theNewBuffer, then the app is surely headed
					//	for trouble in any case.
					fatalError("fail to create theNewBuffer")
				}
				
				theNewBuffer.label = itsBufferLabel + " #" + String(itsCount)
				itsCount += 1
				
				theBuffer = theNewBuffer
			}
			else {
				theBuffer = itsPool.removeLast()
			}

		itsPoolLock.unlock()
		
		return theBuffer
	}
	
	func put(_ buffer: MTLBuffer) {

		itsPoolLock.lock()
			self.itsPool.append(buffer)
		itsPoolLock.unlock()
	}
}
